/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job.process.autodetect.output;

import java.time.Duration;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.MachineLearningField;
import org.elasticsearch.xpack.core.ml.action.PutJobAction;
import org.elasticsearch.xpack.core.ml.action.UpdateJobAction;
import org.elasticsearch.xpack.core.ml.job.config.JobUpdate;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.output.FlushAcknowledgement;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSizeStats;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSnapshot;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.Quantiles;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.TimingStats;
import org.elasticsearch.xpack.core.ml.job.results.AnomalyRecord;
import org.elasticsearch.xpack.core.ml.job.results.Bucket;
import org.elasticsearch.xpack.core.ml.job.results.CategoryDefinition;
import org.elasticsearch.xpack.core.ml.job.results.Forecast;
import org.elasticsearch.xpack.core.ml.job.results.ForecastRequestStats;
import org.elasticsearch.xpack.core.ml.job.results.Influencer;
import org.elasticsearch.xpack.core.ml.job.results.ModelPlot;
import org.elasticsearch.xpack.ml.job.persistence.JobResultsPersister;
import org.elasticsearch.xpack.ml.job.persistence.TimingStatsReporter;
import org.elasticsearch.xpack.ml.job.process.autodetect.AutodetectProcess;
import org.elasticsearch.xpack.ml.job.process.autodetect.output.FlushListener;
import org.elasticsearch.xpack.ml.job.process.normalizer.Renormalizer;
import org.elasticsearch.xpack.ml.job.results.AutodetectResult;
import org.elasticsearch.xpack.ml.notifications.AnomalyDetectionAuditor;

public class AutodetectResultProcessor {
    private static final Logger LOGGER = LogManager.getLogger(AutodetectResultProcessor.class);
    private final Client client;
    private final AnomalyDetectionAuditor auditor;
    private final String jobId;
    private final Renormalizer renormalizer;
    private final JobResultsPersister persister;
    private final AutodetectProcess process;
    private final TimingStatsReporter timingStatsReporter;
    final CountDownLatch completionLatch = new CountDownLatch(1);
    final Semaphore updateModelSnapshotSemaphore = new Semaphore(1);
    private final FlushListener flushListener;
    private volatile boolean processKilled;
    private volatile boolean failed;
    private int bucketCount;
    private final JobResultsPersister.Builder bulkResultsPersister;
    private boolean deleteInterimRequired;
    private volatile ModelSizeStats latestModelSizeStats;

    public AutodetectResultProcessor(Client client, AnomalyDetectionAuditor auditor, String jobId, Renormalizer renormalizer, JobResultsPersister persister, AutodetectProcess process, ModelSizeStats latestModelSizeStats, TimingStats timingStats) {
        this(client, auditor, jobId, renormalizer, persister, process, latestModelSizeStats, timingStats, new FlushListener());
    }

    AutodetectResultProcessor(Client client, AnomalyDetectionAuditor auditor, String jobId, Renormalizer renormalizer, JobResultsPersister persister, AutodetectProcess autodetectProcess, ModelSizeStats latestModelSizeStats, TimingStats timingStats, FlushListener flushListener) {
        this.client = Objects.requireNonNull(client);
        this.auditor = Objects.requireNonNull(auditor);
        this.jobId = Objects.requireNonNull(jobId);
        this.renormalizer = Objects.requireNonNull(renormalizer);
        this.persister = Objects.requireNonNull(persister);
        this.process = Objects.requireNonNull(autodetectProcess);
        this.flushListener = Objects.requireNonNull(flushListener);
        this.latestModelSizeStats = Objects.requireNonNull(latestModelSizeStats);
        this.bulkResultsPersister = persister.bulkPersisterBuilder(jobId);
        this.timingStatsReporter = new TimingStatsReporter(timingStats, this.bulkResultsPersister);
        this.deleteInterimRequired = true;
    }

    public void process() {
        try {
            this.readResults();
            try {
                if (!this.processKilled) {
                    this.timingStatsReporter.finishReporting();
                    this.bulkResultsPersister.executeRequest();
                }
            }
            catch (Exception e) {
                LOGGER.warn((Message)new ParameterizedMessage("[{}] Error persisting autodetect results", (Object)this.jobId), (Throwable)e);
            }
            LOGGER.info("[{}] {} buckets parsed from autodetect output", (Object)this.jobId, (Object)this.bucketCount);
        }
        catch (Exception e) {
            this.failed = true;
            if (this.processKilled) {
                LOGGER.warn("[{}] some results not processed due to the process being killed", (Object)this.jobId);
            } else if (!this.process.isProcessAliveAfterWaiting()) {
                LOGGER.warn("[{}] some results not processed due to the termination of autodetect", (Object)this.jobId);
            } else {
                LOGGER.error((Message)new ParameterizedMessage("[{}] error parsing autodetect output", (Object)this.jobId), (Throwable)e);
            }
        }
        finally {
            this.flushListener.clear();
            this.completionLatch.countDown();
        }
    }

    private void readResults() {
        this.bucketCount = 0;
        try {
            Iterator<AutodetectResult> iterator = this.process.readAutodetectResults();
            while (iterator.hasNext()) {
                try {
                    AutodetectResult result = iterator.next();
                    this.processResult(result);
                    if (result.getBucket() == null) continue;
                    LOGGER.trace("[{}] Bucket number {} parsed from output", (Object)this.jobId, (Object)this.bucketCount);
                }
                catch (Exception e) {
                    if (this.processKilled) {
                        throw e;
                    }
                    if (!this.process.isProcessAliveAfterWaiting()) {
                        throw e;
                    }
                    LOGGER.warn((Message)new ParameterizedMessage("[{}] Error processing autodetect result", (Object)this.jobId), (Throwable)e);
                }
            }
        }
        finally {
            this.process.consumeAndCloseOutputStream();
        }
    }

    public void setProcessKilled() {
        this.processKilled = true;
        this.renormalizer.shutdown();
    }

    void processResult(AutodetectResult result) {
        FlushAcknowledgement flushAcknowledgement;
        Quantiles quantiles;
        IndexResponse indexResponse;
        ModelSnapshot modelSnapshot;
        ModelSizeStats modelSizeStats;
        ForecastRequestStats forecastRequestStats;
        Forecast forecast;
        ModelPlot modelPlot;
        CategoryDefinition categoryDefinition;
        List<Influencer> influencers;
        List<AnomalyRecord> records;
        if (this.processKilled) {
            return;
        }
        Bucket bucket = result.getBucket();
        if (bucket != null) {
            if (this.deleteInterimRequired) {
                LOGGER.trace("[{}] Deleting interim results", (Object)this.jobId);
                this.persister.deleteInterimResults(this.jobId);
                this.deleteInterimRequired = false;
            }
            this.timingStatsReporter.reportBucket(bucket);
            this.bulkResultsPersister.persistBucket(bucket).executeRequest();
            ++this.bucketCount;
        }
        if ((records = result.getRecords()) != null && !records.isEmpty()) {
            this.bulkResultsPersister.persistRecords(records);
        }
        if ((influencers = result.getInfluencers()) != null && !influencers.isEmpty()) {
            this.bulkResultsPersister.persistInfluencers(influencers);
        }
        if ((categoryDefinition = result.getCategoryDefinition()) != null) {
            this.persister.persistCategoryDefinition(categoryDefinition);
        }
        if ((modelPlot = result.getModelPlot()) != null) {
            this.bulkResultsPersister.persistModelPlot(modelPlot);
        }
        if ((forecast = result.getForecast()) != null) {
            this.bulkResultsPersister.persistForecast(forecast);
        }
        if ((forecastRequestStats = result.getForecastRequestStats()) != null) {
            LOGGER.trace("Received Forecast Stats [{}]", (Object)forecastRequestStats.getId());
            this.bulkResultsPersister.persistForecastRequestStats(forecastRequestStats);
            switch (forecastRequestStats.getStatus()) {
                case OK: 
                case STARTED: {
                    break;
                }
                default: {
                    this.bulkResultsPersister.executeRequest();
                }
            }
        }
        if ((modelSizeStats = result.getModelSizeStats()) != null) {
            this.processModelSizeStats(modelSizeStats);
        }
        if ((modelSnapshot = result.getModelSnapshot()) != null && (indexResponse = this.persister.persistModelSnapshot(modelSnapshot, WriteRequest.RefreshPolicy.IMMEDIATE)).getResult() == DocWriteResponse.Result.CREATED) {
            this.updateModelSnapshotOnJob(modelSnapshot);
        }
        if ((quantiles = result.getQuantiles()) != null) {
            LOGGER.debug("[{}] Parsed Quantiles with timestamp {}", (Object)this.jobId, (Object)quantiles.getTimestamp());
            this.persister.persistQuantiles(quantiles);
            this.bulkResultsPersister.executeRequest();
            if (!this.processKilled && this.renormalizer.isEnabled()) {
                this.persister.commitResultWrites(this.jobId);
                LOGGER.debug("[{}] Quantiles queued for renormalization", (Object)this.jobId);
                this.renormalizer.renormalize(quantiles);
            }
        }
        if ((flushAcknowledgement = result.getFlushAcknowledgement()) != null) {
            LOGGER.debug("[{}] Flush acknowledgement parsed from output for ID {}", (Object)this.jobId, (Object)flushAcknowledgement.getId());
            Exception exception = null;
            try {
                this.bulkResultsPersister.executeRequest();
                this.persister.commitResultWrites(this.jobId);
                LOGGER.debug("[{}] Flush acknowledgement sent to listener for ID {}", (Object)this.jobId, (Object)flushAcknowledgement.getId());
            }
            catch (Exception e) {
                LOGGER.error("[" + this.jobId + "] failed to bulk persist results and commit writes during flush acknowledgement for ID " + flushAcknowledgement.getId(), (Throwable)e);
                exception = e;
                throw e;
            }
            finally {
                this.flushListener.acknowledgeFlush(flushAcknowledgement, exception);
            }
            this.deleteInterimRequired = true;
        }
    }

    private void processModelSizeStats(ModelSizeStats modelSizeStats) {
        LOGGER.trace("[{}] Parsed ModelSizeStats: {} / {} / {} / {} / {} / {}", (Object)this.jobId, (Object)modelSizeStats.getModelBytes(), (Object)modelSizeStats.getTotalByFieldCount(), (Object)modelSizeStats.getTotalOverFieldCount(), (Object)modelSizeStats.getTotalPartitionFieldCount(), (Object)modelSizeStats.getBucketAllocationFailuresCount(), (Object)modelSizeStats.getMemoryStatus());
        this.persister.persistModelSizeStats(modelSizeStats);
        this.notifyModelMemoryStatusChange(modelSizeStats);
        this.latestModelSizeStats = modelSizeStats;
    }

    private void notifyModelMemoryStatusChange(ModelSizeStats modelSizeStats) {
        ModelSizeStats.MemoryStatus memoryStatus = modelSizeStats.getMemoryStatus();
        if (memoryStatus != this.latestModelSizeStats.getMemoryStatus()) {
            if (memoryStatus == ModelSizeStats.MemoryStatus.SOFT_LIMIT) {
                this.auditor.warning(this.jobId, Messages.getMessage((String)"Job memory status changed to soft_limit; memory pruning will now be more aggressive"));
            } else if (memoryStatus == ModelSizeStats.MemoryStatus.HARD_LIMIT) {
                if (modelSizeStats.getModelBytesMemoryLimit() == null || modelSizeStats.getModelBytesExceeded() == null) {
                    this.auditor.error(this.jobId, Messages.getMessage((String)"Job memory status changed to hard_limit at {0}; adjust the analysis_limits.model_memory_limit setting to ensure all data is analyzed", (Object[])new Object[]{new ByteSizeValue(modelSizeStats.getModelBytes(), ByteSizeUnit.BYTES).toString()}));
                } else {
                    this.auditor.error(this.jobId, Messages.getMessage((String)"Job memory status changed to hard_limit; job exceeded model memory limit {0} by {1}. Adjust the analysis_limits.model_memory_limit setting to ensure all data is analyzed", (Object[])new Object[]{new ByteSizeValue(modelSizeStats.getModelBytesMemoryLimit().longValue(), ByteSizeUnit.BYTES).toString(), new ByteSizeValue(modelSizeStats.getModelBytesExceeded().longValue(), ByteSizeUnit.BYTES).toString()}));
                }
            }
        }
    }

    protected void updateModelSnapshotOnJob(final ModelSnapshot modelSnapshot) {
        JobUpdate update = new JobUpdate.Builder(this.jobId).setModelSnapshotId(modelSnapshot.getSnapshotId()).build();
        UpdateJobAction.Request updateRequest = UpdateJobAction.Request.internal((String)this.jobId, (JobUpdate)update);
        try {
            this.updateModelSnapshotSemaphore.acquire();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.info("[{}] Interrupted acquiring update model snapshot semaphore", (Object)this.jobId);
            return;
        }
        ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"ml", (ActionType)UpdateJobAction.INSTANCE, (ActionRequest)updateRequest, (ActionListener)new ActionListener<PutJobAction.Response>(){

            public void onResponse(PutJobAction.Response response) {
                AutodetectResultProcessor.this.updateModelSnapshotSemaphore.release();
                LOGGER.debug("[{}] Updated job with model snapshot id [{}]", (Object)AutodetectResultProcessor.this.jobId, (Object)modelSnapshot.getSnapshotId());
            }

            public void onFailure(Exception e) {
                AutodetectResultProcessor.this.updateModelSnapshotSemaphore.release();
                LOGGER.error("[" + AutodetectResultProcessor.this.jobId + "] Failed to update job with new model snapshot id [" + modelSnapshot.getSnapshotId() + "]", (Throwable)e);
            }
        });
    }

    public void awaitCompletion() throws TimeoutException {
        try {
            if (!this.completionLatch.await(MachineLearningField.STATE_PERSIST_RESTORE_TIMEOUT.getMinutes(), TimeUnit.MINUTES)) {
                throw new TimeoutException("Timed out waiting for results processor to complete for job " + this.jobId);
            }
            this.updateModelSnapshotSemaphore.acquire();
            this.updateModelSnapshotSemaphore.release();
            this.waitUntilRenormalizerIsIdle();
            this.persister.commitResultWrites(this.jobId);
            this.persister.commitStateWrites(this.jobId);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.info("[{}] Interrupted waiting for results processor to complete", (Object)this.jobId);
        }
    }

    @Nullable
    public FlushAcknowledgement waitForFlushAcknowledgement(String flushId, Duration timeout) throws Exception {
        return this.failed ? null : this.flushListener.waitForFlush(flushId, timeout);
    }

    public void clearAwaitingFlush(String flushId) {
        this.flushListener.clear(flushId);
    }

    public void waitUntilRenormalizerIsIdle() {
        this.renormalizer.waitUntilIdle();
    }

    public boolean isFailed() {
        return this.failed;
    }

    public ModelSizeStats modelSizeStats() {
        return this.latestModelSizeStats;
    }

    public TimingStats timingStats() {
        return this.timingStatsReporter.getCurrentTimingStats();
    }

    boolean isDeleteInterimRequired() {
        return this.deleteInterimRequired;
    }

    void setDeleteInterimRequired(boolean deleteInterimRequired) {
        this.deleteInterimRequired = deleteInterimRequired;
    }
}

