/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.datafeed;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.persistent.PersistentTaskState;
import org.elasticsearch.persistent.PersistentTasksCustomMetaData;
import org.elasticsearch.persistent.PersistentTasksService;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.MlTasks;
import org.elasticsearch.xpack.core.ml.action.CloseJobAction;
import org.elasticsearch.xpack.core.ml.action.StartDatafeedAction;
import org.elasticsearch.xpack.core.ml.datafeed.DatafeedState;
import org.elasticsearch.xpack.core.ml.job.config.JobState;
import org.elasticsearch.xpack.core.ml.job.config.JobTaskState;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.ml.action.TransportStartDatafeedAction;
import org.elasticsearch.xpack.ml.datafeed.DatafeedJob;
import org.elasticsearch.xpack.ml.datafeed.DatafeedJobBuilder;
import org.elasticsearch.xpack.ml.datafeed.ProblemTracker;
import org.elasticsearch.xpack.ml.job.process.autodetect.AutodetectProcessManager;
import org.elasticsearch.xpack.ml.notifications.AnomalyDetectionAuditor;

public class DatafeedManager {
    private static final Logger logger = LogManager.getLogger(DatafeedManager.class);
    private final Client client;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final Supplier<Long> currentTimeSupplier;
    private final AnomalyDetectionAuditor auditor;
    private final ConcurrentMap<Long, Holder> runningDatafeedsOnThisNode = new ConcurrentHashMap<Long, Holder>();
    private final DatafeedJobBuilder datafeedJobBuilder;
    private final TaskRunner taskRunner = new TaskRunner();
    private final AutodetectProcessManager autodetectProcessManager;

    public DatafeedManager(ThreadPool threadPool, Client client, ClusterService clusterService, DatafeedJobBuilder datafeedJobBuilder, Supplier<Long> currentTimeSupplier, AnomalyDetectionAuditor auditor, AutodetectProcessManager autodetectProcessManager) {
        this.client = Objects.requireNonNull(client);
        this.clusterService = Objects.requireNonNull(clusterService);
        this.threadPool = threadPool;
        this.currentTimeSupplier = Objects.requireNonNull(currentTimeSupplier);
        this.auditor = Objects.requireNonNull(auditor);
        this.datafeedJobBuilder = Objects.requireNonNull(datafeedJobBuilder);
        this.autodetectProcessManager = autodetectProcessManager;
        clusterService.addListener((ClusterStateListener)this.taskRunner);
    }

    public void run(final TransportStartDatafeedAction.DatafeedTask task, final Consumer<Exception> finishHandler) {
        final String datafeedId = task.getDatafeedId();
        ActionListener datafeedJobHandler = ActionListener.wrap(datafeedJob -> {
            Holder holder = new Holder(task, datafeedId, (DatafeedJob)datafeedJob, new ProblemTracker(this.auditor, datafeedJob.getJobId()), finishHandler);
            this.runningDatafeedsOnThisNode.put(task.getAllocationId(), holder);
            task.updatePersistentTaskState((PersistentTaskState)DatafeedState.STARTED, new ActionListener<PersistentTasksCustomMetaData.PersistentTask<?>>(){

                public void onResponse(PersistentTasksCustomMetaData.PersistentTask<?> persistentTask) {
                    DatafeedManager.this.taskRunner.runWhenJobIsOpened(task);
                }

                public void onFailure(Exception e) {
                    if (ExceptionsHelper.unwrapCause((Throwable)e) instanceof ResourceNotFoundException) {
                        logger.info("[{}] Aborting as datafeed has been stopped", (Object)datafeedId);
                    } else {
                        finishHandler.accept(e);
                    }
                }
            });
        }, finishHandler::accept);
        this.datafeedJobBuilder.build(datafeedId, (ActionListener<DatafeedJob>)datafeedJobHandler);
    }

    public void stopDatafeed(TransportStartDatafeedAction.DatafeedTask task, String reason, TimeValue timeout) {
        logger.info("[{}] attempt to stop datafeed [{}] [{}]", (Object)reason, (Object)task.getDatafeedId(), (Object)task.getAllocationId());
        Holder holder = (Holder)this.runningDatafeedsOnThisNode.remove(task.getAllocationId());
        if (holder != null) {
            holder.stop(reason, timeout, null);
        }
    }

    public void stopAllDatafeedsOnThisNode(String reason) {
        int numDatafeeds = this.runningDatafeedsOnThisNode.size();
        if (numDatafeeds != 0) {
            logger.info("Closing [{}] datafeeds, because [{}]", (Object)numDatafeeds, (Object)reason);
            for (Holder holder : this.runningDatafeedsOnThisNode.values()) {
                holder.stop(reason, TimeValue.timeValueSeconds((long)20L), null);
            }
        }
    }

    public void isolateAllDatafeedsOnThisNodeBeforeShutdown() {
        Iterator iter = this.runningDatafeedsOnThisNode.values().iterator();
        while (iter.hasNext()) {
            Holder next = (Holder)iter.next();
            next.isolateDatafeed();
            next.setNodeIsShuttingDown();
            iter.remove();
        }
    }

    public void isolateDatafeed(long allocationId) {
        Holder holder = (Holder)this.runningDatafeedsOnThisNode.get(allocationId);
        if (holder != null) {
            holder.isolateDatafeed();
        }
    }

    private void innerRun(final Holder holder, final long startTime, final Long endTime) {
        holder.cancellable = Scheduler.wrapAsCancellable(this.threadPool.executor("ml_datafeed").submit((Runnable)new AbstractRunnable(){

            public void onFailure(Exception e) {
                logger.error("Failed lookback import for job [" + holder.datafeedJob.getJobId() + "]", (Throwable)e);
                holder.stop("general_lookback_failure", TimeValue.timeValueSeconds((long)20L), e);
            }

            protected void doRun() {
                Long next = null;
                try {
                    next = holder.executeLookBack(startTime, endTime);
                }
                catch (DatafeedJob.ExtractionProblemException e) {
                    if (endTime == null) {
                        next = e.nextDelayInMsSinceEpoch;
                    }
                    holder.problemTracker.reportExtractionProblem(e.getCause().getMessage());
                }
                catch (DatafeedJob.AnalysisProblemException e) {
                    if (endTime == null) {
                        next = e.nextDelayInMsSinceEpoch;
                    }
                    holder.problemTracker.reportAnalysisProblem(e.getCause().getMessage());
                    if (e.shouldStop) {
                        holder.stop("lookback_analysis_error", TimeValue.timeValueSeconds((long)20L), e);
                        return;
                    }
                }
                catch (DatafeedJob.EmptyDataCountException e) {
                    if (endTime == null) {
                        holder.problemTracker.reportEmptyDataCount();
                        next = e.nextDelayInMsSinceEpoch;
                    } else {
                        String lookbackNoDataMsg = Messages.getMessage((String)"Datafeed lookback retrieved no data");
                        logger.warn("[{}] {}", (Object)holder.datafeedJob.getJobId(), (Object)lookbackNoDataMsg);
                        DatafeedManager.this.auditor.warning(holder.datafeedJob.getJobId(), lookbackNoDataMsg);
                    }
                }
                catch (Exception e) {
                    logger.error("Failed lookback import for job [" + holder.datafeedJob.getJobId() + "]", (Throwable)e);
                    holder.stop("general_lookback_failure", TimeValue.timeValueSeconds((long)20L), e);
                    return;
                }
                if (!holder.isIsolated()) {
                    if (next != null) {
                        DatafeedManager.this.doDatafeedRealtime(next, holder.datafeedJob.getJobId(), holder);
                    } else {
                        holder.stop("no_realtime", TimeValue.timeValueSeconds((long)20L), null);
                        holder.problemTracker.finishReport();
                    }
                }
            }
        }));
    }

    void doDatafeedRealtime(long delayInMsSinceEpoch, final String jobId, final Holder holder) {
        if (holder.isRunning() && !holder.isIsolated()) {
            TimeValue delay = this.computeNextDelay(delayInMsSinceEpoch);
            logger.debug("Waiting [{}] before executing next realtime import for job [{}]", (Object)delay, (Object)jobId);
            holder.cancellable = this.threadPool.schedule((Runnable)new AbstractRunnable(){

                public void onFailure(Exception e) {
                    logger.error("Unexpected datafeed failure for job [" + jobId + "] stopping...", (Throwable)e);
                    holder.stop("general_realtime_error", TimeValue.timeValueSeconds((long)20L), e);
                }

                protected void doRun() {
                    long nextDelayInMsSinceEpoch;
                    try {
                        nextDelayInMsSinceEpoch = holder.executeRealTime();
                        holder.problemTracker.reportNonEmptyDataCount();
                    }
                    catch (DatafeedJob.ExtractionProblemException e) {
                        nextDelayInMsSinceEpoch = e.nextDelayInMsSinceEpoch;
                        holder.problemTracker.reportExtractionProblem(e.getCause().getMessage());
                    }
                    catch (DatafeedJob.AnalysisProblemException e) {
                        nextDelayInMsSinceEpoch = e.nextDelayInMsSinceEpoch;
                        holder.problemTracker.reportAnalysisProblem(e.getCause().getMessage());
                        if (e.shouldStop) {
                            holder.stop("realtime_analysis_error", TimeValue.timeValueSeconds((long)20L), e);
                            return;
                        }
                    }
                    catch (DatafeedJob.EmptyDataCountException e) {
                        int emptyDataCount = holder.problemTracker.reportEmptyDataCount();
                        if (!e.haveEverSeenData && holder.shouldStopAfterEmptyData(emptyDataCount)) {
                            logger.warn("Datafeed for [" + jobId + "] has seen no data in [" + emptyDataCount + "] attempts, and never seen any data previously, so stopping...");
                            holder.stop("no_data", TimeValue.timeValueSeconds((long)20L), e, true);
                            return;
                        }
                        nextDelayInMsSinceEpoch = e.nextDelayInMsSinceEpoch;
                    }
                    catch (Exception e) {
                        logger.error("Unexpected datafeed failure for job [" + jobId + "] stopping...", (Throwable)e);
                        holder.stop("general_realtime_error", TimeValue.timeValueSeconds((long)20L), e);
                        return;
                    }
                    holder.problemTracker.finishReport();
                    if (nextDelayInMsSinceEpoch >= 0L) {
                        DatafeedManager.this.doDatafeedRealtime(nextDelayInMsSinceEpoch, jobId, holder);
                    }
                }
            }, delay, "ml_datafeed");
        }
    }

    private String getJobId(TransportStartDatafeedAction.DatafeedTask task) {
        return ((Holder)this.runningDatafeedsOnThisNode.get(task.getAllocationId())).getJobId();
    }

    private JobState getJobState(PersistentTasksCustomMetaData tasks, TransportStartDatafeedAction.DatafeedTask datafeedTask) {
        return MlTasks.getJobStateModifiedForReassignments((String)this.getJobId(datafeedTask), (PersistentTasksCustomMetaData)tasks);
    }

    private boolean jobHasOpenAutodetectCommunicator(PersistentTasksCustomMetaData tasks, TransportStartDatafeedAction.DatafeedTask datafeedTask) {
        PersistentTasksCustomMetaData.PersistentTask jobTask = MlTasks.getJobTask((String)this.getJobId(datafeedTask), (PersistentTasksCustomMetaData)tasks);
        if (jobTask == null) {
            return false;
        }
        JobTaskState state = (JobTaskState)jobTask.getState();
        if (state == null || state.isStatusStale(jobTask)) {
            return false;
        }
        return this.autodetectProcessManager.hasOpenAutodetectCommunicator(jobTask.getAllocationId());
    }

    private TimeValue computeNextDelay(long next) {
        return new TimeValue(Math.max(1L, next - this.currentTimeSupplier.get()));
    }

    boolean isRunning(long allocationId) {
        return this.runningDatafeedsOnThisNode.containsKey(allocationId);
    }

    private class TaskRunner
    implements ClusterStateListener {
        private final List<TransportStartDatafeedAction.DatafeedTask> tasksToRun = new CopyOnWriteArrayList<TransportStartDatafeedAction.DatafeedTask>();

        private TaskRunner() {
        }

        private void runWhenJobIsOpened(TransportStartDatafeedAction.DatafeedTask datafeedTask) {
            ClusterState clusterState = DatafeedManager.this.clusterService.state();
            PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)clusterState.getMetaData().custom("persistent_tasks");
            if (DatafeedManager.this.getJobState(tasks, datafeedTask) == JobState.OPENED && DatafeedManager.this.jobHasOpenAutodetectCommunicator(tasks, datafeedTask)) {
                this.runTask(datafeedTask);
            } else {
                logger.info("Datafeed [{}] is waiting for job [{}] to be opened", (Object)datafeedTask.getDatafeedId(), (Object)DatafeedManager.this.getJobId(datafeedTask));
                this.tasksToRun.add(datafeedTask);
            }
        }

        private void runTask(TransportStartDatafeedAction.DatafeedTask task) {
            try (ThreadContext.StoredContext ignore = DatafeedManager.this.threadPool.getThreadContext().stashContext();){
                DatafeedManager.this.innerRun((Holder)DatafeedManager.this.runningDatafeedsOnThisNode.get(task.getAllocationId()), task.getDatafeedStartTime(), task.getEndTime());
            }
        }

        public void clusterChanged(ClusterChangedEvent event) {
            PersistentTasksCustomMetaData currentTasks;
            if (this.tasksToRun.isEmpty() || !event.metaDataChanged()) {
                return;
            }
            PersistentTasksCustomMetaData previousTasks = (PersistentTasksCustomMetaData)event.previousState().getMetaData().custom("persistent_tasks");
            if (Objects.equals(previousTasks, currentTasks = (PersistentTasksCustomMetaData)event.state().getMetaData().custom("persistent_tasks"))) {
                return;
            }
            ArrayList<TransportStartDatafeedAction.DatafeedTask> remainingTasks = new ArrayList<TransportStartDatafeedAction.DatafeedTask>();
            for (TransportStartDatafeedAction.DatafeedTask datafeedTask : this.tasksToRun) {
                if (!DatafeedManager.this.runningDatafeedsOnThisNode.containsKey(datafeedTask.getAllocationId())) continue;
                JobState jobState = DatafeedManager.this.getJobState(currentTasks, datafeedTask);
                if (jobState == JobState.OPENING || !DatafeedManager.this.jobHasOpenAutodetectCommunicator(currentTasks, datafeedTask)) {
                    remainingTasks.add(datafeedTask);
                    continue;
                }
                if (jobState == JobState.OPENED) {
                    this.runTask(datafeedTask);
                    continue;
                }
                logger.warn("Datafeed [{}] is stopping because job [{}] state is [{}]", (Object)datafeedTask.getDatafeedId(), (Object)DatafeedManager.this.getJobId(datafeedTask), (Object)jobState);
                datafeedTask.stop("job_never_opened", TimeValue.timeValueSeconds((long)20L));
            }
            this.tasksToRun.retainAll(remainingTasks);
        }
    }

    public class Holder {
        private final TransportStartDatafeedAction.DatafeedTask task;
        private final long allocationId;
        private final String datafeedId;
        private final ReentrantLock datafeedJobLock = new ReentrantLock(true);
        private final DatafeedJob datafeedJob;
        private final boolean defaultAutoCloseJob;
        private final ProblemTracker problemTracker;
        private final Consumer<Exception> finishHandler;
        volatile Scheduler.Cancellable cancellable;
        private volatile boolean isNodeShuttingDown;

        Holder(TransportStartDatafeedAction.DatafeedTask task, String datafeedId, DatafeedJob datafeedJob, ProblemTracker problemTracker, Consumer<Exception> finishHandler) {
            this.task = task;
            this.allocationId = task.getAllocationId();
            this.datafeedId = datafeedId;
            this.datafeedJob = datafeedJob;
            this.defaultAutoCloseJob = task.isLookbackOnly();
            this.problemTracker = problemTracker;
            this.finishHandler = finishHandler;
        }

        boolean shouldStopAfterEmptyData(int emptyDataCount) {
            Integer emptyDataCountToStopAt = this.datafeedJob.getMaxEmptySearches();
            return emptyDataCountToStopAt != null && emptyDataCount >= emptyDataCountToStopAt;
        }

        String getJobId() {
            return this.datafeedJob.getJobId();
        }

        boolean isRunning() {
            return this.datafeedJob.isRunning();
        }

        boolean isIsolated() {
            return this.datafeedJob.isIsolated();
        }

        public void stop(String source, TimeValue timeout, Exception e) {
            this.stop(source, timeout, e, this.defaultAutoCloseJob);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void stop(String source, TimeValue timeout, Exception e, boolean autoCloseJob) {
            block14: {
                block15: {
                    if (this.isNodeShuttingDown) {
                        return;
                    }
                    logger.info("[{}] attempt to stop datafeed [{}] for job [{}]", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId());
                    if (!this.datafeedJob.stop()) break block15;
                    boolean acquired = false;
                    try {
                        logger.info("[{}] try lock [{}] to stop datafeed [{}] for job [{}]...", (Object)source, (Object)timeout, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId());
                        acquired = this.datafeedJobLock.tryLock(timeout.millis(), TimeUnit.MILLISECONDS);
                    }
                    catch (InterruptedException e1) {
                        try {
                            Thread.currentThread().interrupt();
                        }
                        catch (Throwable throwable) {
                            logger.info("[{}] stopping datafeed [{}] for job [{}], acquired [{}]...", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)acquired);
                            DatafeedManager.this.runningDatafeedsOnThisNode.remove(this.allocationId);
                            if (this.cancellable != null) {
                                this.cancellable.cancel();
                            }
                            DatafeedManager.this.auditor.info(this.datafeedJob.getJobId(), Messages.getMessage((String)(this.isIsolated() ? "Datafeed isolated" : "Datafeed stopped")));
                            this.datafeedJob.finishReportingTimingStats();
                            this.finishHandler.accept(e);
                            logger.info("[{}] datafeed [{}] for job [{}] has been stopped{}", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)(acquired ? "" : ", but there may be pending tasks as the timeout [" + timeout.getStringRep() + "] expired"));
                            if (autoCloseJob && !this.isIsolated()) {
                                this.closeJob();
                            }
                            if (acquired) {
                                this.datafeedJobLock.unlock();
                            }
                            throw throwable;
                        }
                        logger.info("[{}] stopping datafeed [{}] for job [{}], acquired [{}]...", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)acquired);
                        DatafeedManager.this.runningDatafeedsOnThisNode.remove(this.allocationId);
                        if (this.cancellable != null) {
                            this.cancellable.cancel();
                        }
                        DatafeedManager.this.auditor.info(this.datafeedJob.getJobId(), Messages.getMessage((String)(this.isIsolated() ? "Datafeed isolated" : "Datafeed stopped")));
                        this.datafeedJob.finishReportingTimingStats();
                        this.finishHandler.accept(e);
                        logger.info("[{}] datafeed [{}] for job [{}] has been stopped{}", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)(acquired ? "" : ", but there may be pending tasks as the timeout [" + timeout.getStringRep() + "] expired"));
                        if (autoCloseJob && !this.isIsolated()) {
                            this.closeJob();
                        }
                        if (acquired) {
                            this.datafeedJobLock.unlock();
                        }
                        break block14;
                    }
                    logger.info("[{}] stopping datafeed [{}] for job [{}], acquired [{}]...", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)acquired);
                    DatafeedManager.this.runningDatafeedsOnThisNode.remove(this.allocationId);
                    if (this.cancellable != null) {
                        this.cancellable.cancel();
                    }
                    DatafeedManager.this.auditor.info(this.datafeedJob.getJobId(), Messages.getMessage((String)(this.isIsolated() ? "Datafeed isolated" : "Datafeed stopped")));
                    this.datafeedJob.finishReportingTimingStats();
                    this.finishHandler.accept(e);
                    logger.info("[{}] datafeed [{}] for job [{}] has been stopped{}", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId(), (Object)(acquired ? "" : ", but there may be pending tasks as the timeout [" + timeout.getStringRep() + "] expired"));
                    if (autoCloseJob && !this.isIsolated()) {
                        this.closeJob();
                    }
                    if (acquired) {
                        this.datafeedJobLock.unlock();
                    }
                    break block14;
                }
                logger.info("[{}] datafeed [{}] for job [{}] was already stopped", (Object)source, (Object)this.datafeedId, (Object)this.datafeedJob.getJobId());
            }
        }

        public void isolateDatafeed() {
            this.datafeedJob.isolate();
        }

        public void setNodeIsShuttingDown() {
            this.isNodeShuttingDown = true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Long executeLookBack(long startTime, Long endTime) throws Exception {
            this.datafeedJobLock.lock();
            try {
                if (this.isRunning() && !this.isIsolated()) {
                    Long l = this.datafeedJob.runLookBack(startTime, endTime);
                    return l;
                }
                Long l = null;
                return l;
            }
            finally {
                this.datafeedJobLock.unlock();
            }
        }

        private long executeRealTime() throws Exception {
            this.datafeedJobLock.lock();
            try {
                if (this.isRunning() && !this.isIsolated()) {
                    long l = this.datafeedJob.runRealtime();
                    return l;
                }
                long l = -1L;
                return l;
            }
            finally {
                this.datafeedJobLock.unlock();
            }
        }

        private void closeJob() {
            ClusterState clusterState = DatafeedManager.this.clusterService.state();
            PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)clusterState.getMetaData().custom("persistent_tasks");
            JobState jobState = MlTasks.getJobState((String)this.getJobId(), (PersistentTasksCustomMetaData)tasks);
            if (jobState != JobState.OPENED) {
                logger.debug("[{}] No need to auto-close job as job state is [{}]", (Object)this.getJobId(), (Object)jobState);
                return;
            }
            this.task.waitForPersistentTask(Objects::isNull, TimeValue.timeValueSeconds((long)20L), (PersistentTasksService.WaitForPersistentTaskListener)new PersistentTasksService.WaitForPersistentTaskListener<StartDatafeedAction.DatafeedParams>(){

                public void onResponse(PersistentTasksCustomMetaData.PersistentTask<StartDatafeedAction.DatafeedParams> persistentTask) {
                    CloseJobAction.Request closeJobRequest = new CloseJobAction.Request(Holder.this.getJobId());
                    closeJobRequest.setLocal(true);
                    ClientHelper.executeAsyncWithOrigin((Client)DatafeedManager.this.client, (String)"ml", (ActionType)CloseJobAction.INSTANCE, (ActionRequest)closeJobRequest, (ActionListener)new ActionListener<CloseJobAction.Response>(){

                        public void onResponse(CloseJobAction.Response response) {
                            if (!response.isClosed()) {
                                logger.error("[{}] job close action was not acknowledged", (Object)Holder.this.getJobId());
                            }
                        }

                        public void onFailure(Exception e) {
                            if (e instanceof ElasticsearchStatusException && ((ElasticsearchStatusException)e).status() == RestStatus.CONFLICT) {
                                logger.debug("[{}] {}", (Object)Holder.this.getJobId(), (Object)e.getMessage());
                            } else {
                                logger.error("[" + Holder.this.getJobId() + "] failed to auto-close job", (Throwable)e);
                            }
                        }
                    });
                }

                public void onFailure(Exception e) {
                    logger.error("Failed to remove datafeed persistent task - will not auto close job [" + Holder.this.getJobId() + "]", (Throwable)e);
                }
            });
        }
    }
}

