/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.blobstore;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.RateLimiter;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.StepListener;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.RepositoryMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.CheckedRunnable;
import org.elasticsearch.common.Numbers;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.fs.FsBlobContainer;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.compress.NotXContentException;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.store.InputStreamIndexInput;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.LoggingDeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardRestoreFailedException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotFailedException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotStatus;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshots;
import org.elasticsearch.index.snapshots.blobstore.RateLimitingInputStream;
import org.elasticsearch.index.snapshots.blobstore.SlicedInputStream;
import org.elasticsearch.index.snapshots.blobstore.SnapshotFiles;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.repositories.IndexId;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryCleanupResult;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryVerificationException;
import org.elasticsearch.repositories.blobstore.ChecksumBlobStoreFormat;
import org.elasticsearch.repositories.blobstore.FileRestoreContext;
import org.elasticsearch.snapshots.SnapshotCreationException;
import org.elasticsearch.snapshots.SnapshotException;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.snapshots.SnapshotMissingException;
import org.elasticsearch.snapshots.SnapshotShardFailure;
import org.elasticsearch.threadpool.ThreadPool;

public abstract class BlobStoreRepository
extends AbstractLifecycleComponent
implements Repository {
    private static final Logger logger = LogManager.getLogger(BlobStoreRepository.class);
    protected final RepositoryMetaData metadata;
    protected final NamedXContentRegistry namedXContentRegistry;
    protected final ThreadPool threadPool;
    private static final int BUFFER_SIZE = 4096;
    public static final String SNAPSHOT_PREFIX = "snap-";
    public static final String SNAPSHOT_CODEC = "snapshot";
    public static final String INDEX_FILE_PREFIX = "index-";
    public static final String INDEX_LATEST_BLOB = "index.latest";
    private static final String TESTS_FILE = "tests-";
    public static final String METADATA_PREFIX = "meta-";
    public static final String METADATA_NAME_FORMAT = "meta-%s.dat";
    private static final String METADATA_CODEC = "metadata";
    private static final String INDEX_METADATA_CODEC = "index-metadata";
    public static final String SNAPSHOT_NAME_FORMAT = "snap-%s.dat";
    private static final String SNAPSHOT_INDEX_PREFIX = "index-";
    private static final String SNAPSHOT_INDEX_NAME_FORMAT = "index-%s";
    private static final String SNAPSHOT_INDEX_CODEC = "snapshots";
    private static final String DATA_BLOB_PREFIX = "__";
    private final boolean compress;
    private final RateLimiter snapshotRateLimiter;
    private final RateLimiter restoreRateLimiter;
    private final CounterMetric snapshotRateLimitingTimeInNanos = new CounterMetric();
    private final CounterMetric restoreRateLimitingTimeInNanos = new CounterMetric();
    private ChecksumBlobStoreFormat<MetaData> globalMetaDataFormat;
    private ChecksumBlobStoreFormat<IndexMetaData> indexMetaDataFormat;
    protected ChecksumBlobStoreFormat<SnapshotInfo> snapshotFormat;
    private final boolean readOnly;
    private final ChecksumBlobStoreFormat<BlobStoreIndexShardSnapshot> indexShardSnapshotFormat;
    private final ChecksumBlobStoreFormat<BlobStoreIndexShardSnapshots> indexShardSnapshotsFormat;
    private final Object lock = new Object();
    private final SetOnce<BlobContainer> blobContainer = new SetOnce();
    private final SetOnce<BlobStore> blobStore = new SetOnce();
    protected final AtomicLong latestKnownRepoGen = new AtomicLong(-1L);

    protected BlobStoreRepository(RepositoryMetaData metadata, boolean compress, NamedXContentRegistry namedXContentRegistry, ThreadPool threadPool) {
        this.compress = compress;
        this.metadata = metadata;
        this.namedXContentRegistry = namedXContentRegistry;
        this.threadPool = threadPool;
        this.snapshotRateLimiter = this.getRateLimiter(metadata.settings(), "max_snapshot_bytes_per_sec", new ByteSizeValue(40L, ByteSizeUnit.MB));
        this.restoreRateLimiter = this.getRateLimiter(metadata.settings(), "max_restore_bytes_per_sec", new ByteSizeValue(40L, ByteSizeUnit.MB));
        this.readOnly = metadata.settings().getAsBoolean("readonly", false);
        this.indexShardSnapshotFormat = new ChecksumBlobStoreFormat(SNAPSHOT_CODEC, SNAPSHOT_NAME_FORMAT, BlobStoreIndexShardSnapshot::fromXContent, namedXContentRegistry, compress);
        this.indexShardSnapshotsFormat = new ChecksumBlobStoreFormat(SNAPSHOT_INDEX_CODEC, SNAPSHOT_INDEX_NAME_FORMAT, BlobStoreIndexShardSnapshots::fromXContent, namedXContentRegistry, compress);
    }

    @Override
    protected void doStart() {
        ByteSizeValue chunkSize = this.chunkSize();
        if (chunkSize != null && chunkSize.getBytes() <= 0L) {
            throw new IllegalArgumentException("the chunk size cannot be negative: [" + chunkSize + "]");
        }
        this.globalMetaDataFormat = new ChecksumBlobStoreFormat(METADATA_CODEC, METADATA_NAME_FORMAT, MetaData::fromXContent, this.namedXContentRegistry, this.compress);
        this.indexMetaDataFormat = new ChecksumBlobStoreFormat(INDEX_METADATA_CODEC, METADATA_NAME_FORMAT, IndexMetaData::fromXContent, this.namedXContentRegistry, this.compress);
        this.snapshotFormat = new ChecksumBlobStoreFormat(SNAPSHOT_CODEC, SNAPSHOT_NAME_FORMAT, SnapshotInfo::fromXContentInternal, this.namedXContentRegistry, this.compress);
    }

    @Override
    protected void doStop() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void doClose() {
        BlobStore store;
        Object object = this.lock;
        synchronized (object) {
            store = (BlobStore)this.blobStore.get();
        }
        if (store != null) {
            try {
                store.close();
            }
            catch (Exception t) {
                logger.warn("cannot close blob store", (Throwable)t);
            }
        }
    }

    public ThreadPool threadPool() {
        return this.threadPool;
    }

    BlobContainer getBlobContainer() {
        return (BlobContainer)this.blobContainer.get();
    }

    protected BlobStore getBlobStore() {
        return (BlobStore)this.blobStore.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected BlobContainer blobContainer() {
        this.assertSnapshotOrGenericThread();
        BlobContainer blobContainer = (BlobContainer)this.blobContainer.get();
        if (blobContainer == null) {
            Object object = this.lock;
            synchronized (object) {
                blobContainer = (BlobContainer)this.blobContainer.get();
                if (blobContainer == null) {
                    blobContainer = this.blobStore().blobContainer(this.basePath());
                    this.blobContainer.set((Object)blobContainer);
                }
            }
        }
        return blobContainer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BlobStore blobStore() {
        this.assertSnapshotOrGenericThread();
        BlobStore store = (BlobStore)this.blobStore.get();
        if (store == null) {
            Object object = this.lock;
            synchronized (object) {
                store = (BlobStore)this.blobStore.get();
                if (store == null) {
                    if (!this.lifecycle.started()) {
                        throw new RepositoryException(this.metadata.name(), "repository is not in started state");
                    }
                    try {
                        store = this.createBlobStore();
                    }
                    catch (RepositoryException e) {
                        throw e;
                    }
                    catch (Exception e) {
                        throw new RepositoryException(this.metadata.name(), "cannot create blob store", e);
                    }
                    this.blobStore.set((Object)store);
                }
            }
        }
        return store;
    }

    protected abstract BlobStore createBlobStore() throws Exception;

    public abstract BlobPath basePath();

    protected final boolean isCompress() {
        return this.compress;
    }

    protected ByteSizeValue chunkSize() {
        return null;
    }

    @Override
    public RepositoryMetaData getMetadata() {
        return this.metadata;
    }

    @Override
    public void initializeSnapshot(SnapshotId snapshotId, List<IndexId> indices, MetaData clusterMetaData) {
        try {
            this.globalMetaDataFormat.write(clusterMetaData, this.blobContainer(), snapshotId.getUUID(), true);
            for (IndexId index : indices) {
                this.indexMetaDataFormat.write(clusterMetaData.index(index.getName()), this.indexContainer(index), snapshotId.getUUID(), true);
            }
        }
        catch (IOException ex) {
            throw new SnapshotCreationException(this.metadata.name(), snapshotId, (Throwable)ex);
        }
    }

    @Override
    public void deleteSnapshot(SnapshotId snapshotId, long repositoryStateId, ActionListener<Void> listener) {
        if (this.isReadOnly()) {
            listener.onFailure(new RepositoryException(this.metadata.name(), "cannot delete snapshot from a readonly repository"));
        } else {
            try {
                Map<String, BlobMetaData> rootBlobs = this.blobContainer().listBlobs();
                RepositoryData repositoryData = this.safeRepositoryData(repositoryStateId, rootBlobs);
                Map<String, BlobContainer> foundIndices = this.blobStore().blobContainer(this.indicesPath()).children();
                this.doDeleteShardSnapshots(snapshotId, repositoryStateId, foundIndices, rootBlobs, repositoryData, listener);
            }
            catch (Exception ex) {
                listener.onFailure(new RepositoryException(this.metadata.name(), "failed to delete snapshot [" + snapshotId + "]", ex));
            }
        }
    }

    private RepositoryData safeRepositoryData(long repositoryStateId, Map<String, BlobMetaData> rootBlobs) {
        long generation = this.latestGeneration(rootBlobs.keySet());
        long genToLoad = this.latestKnownRepoGen.updateAndGet(known -> Math.max(known, repositoryStateId));
        if (genToLoad > generation) {
            logger.debug("Determined repository's generation from its contents to [" + generation + "] but current generation is at least [" + genToLoad + "]");
        }
        if (genToLoad != repositoryStateId) {
            throw new RepositoryException(this.metadata.name(), "concurrent modification of the index-N file, expected current generation [" + repositoryStateId + "], actual current generation [" + genToLoad + "]");
        }
        return this.getRepositoryData(genToLoad);
    }

    private void doDeleteShardSnapshots(SnapshotId snapshotId, long repositoryStateId, Map<String, BlobContainer> foundIndices, Map<String, BlobMetaData> rootBlobs, RepositoryData repositoryData, ActionListener<Void> listener) throws IOException {
        RepositoryData updatedRepositoryData = repositoryData.removeSnapshot(snapshotId);
        this.writeIndexGen(updatedRepositoryData, repositoryStateId);
        GroupedActionListener afterCleanupsListener = new GroupedActionListener(ActionListener.wrap(() -> listener.onResponse(null)), 2);
        this.threadPool.executor(SNAPSHOT_CODEC).execute(ActionRunnable.wrap(afterCleanupsListener, l -> this.cleanupStaleBlobs(foundIndices, rootBlobs, updatedRepositoryData, ActionListener.map(l, ignored -> null))));
        this.deleteIndices(updatedRepositoryData, repositoryData.indicesToUpdateAfterRemovingSnapshot(snapshotId), snapshotId, ActionListener.runAfter(ActionListener.wrap(deleteResults -> {
            String basePath = this.basePath().buildAsString();
            int basePathLen = basePath.length();
            this.blobContainer().deleteBlobsIgnoringIfNotExists(Stream.concat(deleteResults.stream().flatMap(shardResult -> {
                String shardPath = this.shardContainer(((ShardSnapshotMetaDeleteResult)shardResult).indexId, ((ShardSnapshotMetaDeleteResult)shardResult).shardId).path().buildAsString();
                return ((ShardSnapshotMetaDeleteResult)shardResult).blobsToDelete.stream().map(blob -> shardPath + blob);
            }), deleteResults.stream().map(shardResult -> ((ShardSnapshotMetaDeleteResult)shardResult).indexId).distinct().map(indexId -> this.indexContainer((IndexId)indexId).path().buildAsString() + this.globalMetaDataFormat.blobName(snapshotId.getUUID()))).map(absolutePath -> {
                assert (absolutePath.startsWith(basePath));
                return absolutePath.substring(basePathLen);
            }).collect(Collectors.toList()));
        }, e -> logger.warn(() -> new ParameterizedMessage("[{}] Failed to delete some blobs during snapshot delete", (Object)snapshotId), (Throwable)e)), () -> afterCleanupsListener.onResponse(null)));
    }

    private void cleanupStaleBlobs(Map<String, BlobContainer> foundIndices, Map<String, BlobMetaData> rootBlobs, RepositoryData newRepoData, ActionListener<DeleteResult> listener) {
        GroupedActionListener groupedListener = new GroupedActionListener(ActionListener.wrap(deleteResults -> {
            DeleteResult deleteResult = DeleteResult.ZERO;
            for (DeleteResult result : deleteResults) {
                deleteResult = deleteResult.add(result);
            }
            listener.onResponse(deleteResult);
        }, listener::onFailure), 2);
        ExecutorService executor = this.threadPool.executor(SNAPSHOT_CODEC);
        executor.execute(ActionRunnable.supply(groupedListener, () -> {
            List<String> deletedBlobs = this.cleanupStaleRootFiles(this.staleRootBlobs(newRepoData, rootBlobs.keySet()));
            return new DeleteResult(deletedBlobs.size(), deletedBlobs.stream().mapToLong(name -> ((BlobMetaData)rootBlobs.get(name)).length()).sum());
        }));
        Set survivingIndexIds = newRepoData.getIndices().values().stream().map(IndexId::getId).collect(Collectors.toSet());
        executor.execute(ActionRunnable.supply(groupedListener, () -> this.cleanupStaleIndices(foundIndices, survivingIndexIds)));
    }

    public void cleanup(long repositoryStateId, ActionListener<RepositoryCleanupResult> listener) {
        try {
            if (this.isReadOnly()) {
                throw new RepositoryException(this.metadata.name(), "cannot run cleanup on readonly repository");
            }
            Map<String, BlobMetaData> rootBlobs = this.blobContainer().listBlobs();
            RepositoryData repositoryData = this.safeRepositoryData(repositoryStateId, rootBlobs);
            Map<String, BlobContainer> foundIndices = this.blobStore().blobContainer(this.indicesPath()).children();
            Set survivingIndexIds = repositoryData.getIndices().values().stream().map(IndexId::getId).collect(Collectors.toSet());
            List<String> staleRootBlobs = this.staleRootBlobs(repositoryData, rootBlobs.keySet());
            if (survivingIndexIds.equals(foundIndices.keySet()) && staleRootBlobs.isEmpty()) {
                listener.onResponse(new RepositoryCleanupResult(DeleteResult.ZERO));
            } else {
                this.writeIndexGen(repositoryData, repositoryStateId);
                this.cleanupStaleBlobs(foundIndices, rootBlobs, repositoryData, ActionListener.map(listener, RepositoryCleanupResult::new));
            }
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private List<String> staleRootBlobs(RepositoryData repositoryData, Set<String> rootBlobNames) {
        Set allSnapshotIds = repositoryData.getSnapshotIds().stream().map(SnapshotId::getUUID).collect(Collectors.toSet());
        return rootBlobNames.stream().filter(blob -> {
            if (FsBlobContainer.isTempBlobName(blob)) {
                return true;
            }
            if (blob.endsWith(".dat")) {
                String foundUUID;
                if (blob.startsWith(SNAPSHOT_PREFIX)) {
                    foundUUID = blob.substring(SNAPSHOT_PREFIX.length(), blob.length() - ".dat".length());
                    assert (this.snapshotFormat.blobName(foundUUID).equals(blob));
                } else if (blob.startsWith(METADATA_PREFIX)) {
                    foundUUID = blob.substring(METADATA_PREFIX.length(), blob.length() - ".dat".length());
                    assert (this.globalMetaDataFormat.blobName(foundUUID).equals(blob));
                } else {
                    return false;
                }
                return !allSnapshotIds.contains(foundUUID);
            }
            return false;
        }).collect(Collectors.toList());
    }

    private List<String> cleanupStaleRootFiles(List<String> blobsToDelete) {
        if (blobsToDelete.isEmpty()) {
            return blobsToDelete;
        }
        try {
            logger.info("[{}] Found stale root level blobs {}. Cleaning them up", (Object)this.metadata.name(), blobsToDelete);
            this.blobContainer().deleteBlobsIgnoringIfNotExists(blobsToDelete);
            return blobsToDelete;
        }
        catch (IOException e) {
            logger.warn(() -> new ParameterizedMessage("[{}] The following blobs are no longer part of any snapshot [{}] but failed to remove them", (Object)this.metadata.name(), (Object)blobsToDelete), (Throwable)e);
        }
        catch (Exception e) {
            assert (false) : e;
            logger.warn((Message)new ParameterizedMessage("[{}] Exception during cleanup of root level blobs", (Object)this.metadata.name()), (Throwable)e);
        }
        return Collections.emptyList();
    }

    private DeleteResult cleanupStaleIndices(Map<String, BlobContainer> foundIndices, Set<String> survivingIndexIds) {
        DeleteResult deleteResult = DeleteResult.ZERO;
        try {
            for (Map.Entry<String, BlobContainer> indexEntry : foundIndices.entrySet()) {
                String indexSnId = indexEntry.getKey();
                try {
                    if (survivingIndexIds.contains(indexSnId)) continue;
                    logger.debug("[{}] Found stale index [{}]. Cleaning it up", (Object)this.metadata.name(), (Object)indexSnId);
                    deleteResult = deleteResult.add(indexEntry.getValue().delete());
                    logger.debug("[{}] Cleaned up stale index [{}]", (Object)this.metadata.name(), (Object)indexSnId);
                }
                catch (IOException e) {
                    logger.warn(() -> new ParameterizedMessage("[{}] index {} is no longer part of any snapshots in the repository, but failed to clean up their index folders", (Object)this.metadata.name(), (Object)indexSnId), (Throwable)e);
                }
            }
        }
        catch (Exception e) {
            assert (false) : e;
            logger.warn((Message)new ParameterizedMessage("[{}] Exception during cleanup of stale indices", (Object)this.metadata.name()), (Throwable)e);
        }
        return deleteResult;
    }

    private void deleteIndices(final RepositoryData repositoryData, List<IndexId> indices, final SnapshotId snapshotId, ActionListener<Collection<ShardSnapshotMetaDeleteResult>> listener) {
        if (indices.isEmpty()) {
            listener.onResponse(Collections.emptyList());
            return;
        }
        GroupedActionListener deleteIndexMetaDataListener = new GroupedActionListener(ActionListener.map(listener, res -> res.stream().flatMap(Collection::stream).collect(Collectors.toList())), indices.size());
        ExecutorService executor = this.threadPool.executor(SNAPSHOT_CODEC);
        for (final IndexId indexId : indices) {
            executor.execute(ActionRunnable.wrap(deleteIndexMetaDataListener, deleteIdxMetaListener -> {
                IndexMetaData indexMetaData;
                try {
                    indexMetaData = this.getSnapshotIndexMetaData(snapshotId, indexId);
                }
                catch (Exception ex) {
                    logger.warn(() -> new ParameterizedMessage("[{}] [{}] failed to read metadata for index", (Object)snapshotId, (Object)indexId.getName()), (Throwable)ex);
                    deleteIdxMetaListener.onResponse(null);
                    return;
                }
                int shardCount = indexMetaData.getNumberOfShards();
                assert (shardCount > 0) : "index did not have positive shard count, get [" + shardCount + "]";
                final GroupedActionListener allShardsListener = new GroupedActionListener(deleteIdxMetaListener, shardCount);
                Index index = indexMetaData.getIndex();
                for (int shardId = 0; shardId < indexMetaData.getNumberOfShards(); ++shardId) {
                    final ShardId shard = new ShardId(index, shardId);
                    executor.execute(new AbstractRunnable(){

                        @Override
                        protected void doRun() throws Exception {
                            allShardsListener.onResponse(BlobStoreRepository.this.deleteShardSnapshot(repositoryData, indexId, shard, snapshotId));
                        }

                        @Override
                        public void onFailure(Exception ex) {
                            logger.warn(() -> new ParameterizedMessage("[{}] failed to delete shard data for shard [{}][{}]", new Object[]{snapshotId, indexId.getName(), shard.id()}), (Throwable)ex);
                            allShardsListener.onResponse(null);
                        }
                    });
                }
            }));
        }
    }

    @Override
    public void finalizeSnapshot(SnapshotId snapshotId, List<IndexId> indices, long startTime, String failure, int totalShards, List<SnapshotShardFailure> shardFailures, long repositoryStateId, boolean includeGlobalState, MetaData clusterMetaData, Map<String, Object> userMetadata, ActionListener<SnapshotInfo> listener) {
        GroupedActionListener allMetaListener = new GroupedActionListener(ActionListener.wrap(snapshotInfos -> {
            assert (snapshotInfos.size() == 1) : "Should have only received a single SnapshotInfo but received " + snapshotInfos;
            SnapshotInfo snapshotInfo = (SnapshotInfo)snapshotInfos.iterator().next();
            this.writeIndexGen(this.getRepositoryData().addSnapshot(snapshotId, snapshotInfo.state(), indices), repositoryStateId);
            listener.onResponse(snapshotInfo);
        }, e -> listener.onFailure(new SnapshotException(this.metadata.name(), snapshotId, "failed to update snapshot in repository", (Throwable)e))), 2 + indices.size());
        ExecutorService executor = this.threadPool.executor(SNAPSHOT_CODEC);
        executor.execute(ActionRunnable.run(allMetaListener, (CheckedRunnable<Exception>)((CheckedRunnable)() -> this.globalMetaDataFormat.write(clusterMetaData, this.blobContainer(), snapshotId.getUUID(), false))));
        for (IndexId index : indices) {
            executor.execute(ActionRunnable.run(allMetaListener, (CheckedRunnable<Exception>)((CheckedRunnable)() -> this.indexMetaDataFormat.write(clusterMetaData.index(index.getName()), this.indexContainer(index), snapshotId.getUUID(), false))));
        }
        executor.execute(ActionRunnable.supply(allMetaListener, () -> {
            SnapshotInfo snapshotInfo = new SnapshotInfo(snapshotId, indices.stream().map(IndexId::getName).collect(Collectors.toList()), startTime, failure, this.threadPool.absoluteTimeInMillis(), totalShards, shardFailures, includeGlobalState, userMetadata);
            this.snapshotFormat.write(snapshotInfo, this.blobContainer(), snapshotId.getUUID(), false);
            return snapshotInfo;
        }));
    }

    @Override
    public SnapshotInfo getSnapshotInfo(SnapshotId snapshotId) {
        try {
            return this.snapshotFormat.read(this.blobContainer(), snapshotId.getUUID());
        }
        catch (NoSuchFileException ex) {
            throw new SnapshotMissingException(this.metadata.name(), snapshotId, (Throwable)ex);
        }
        catch (IOException | NotXContentException ex) {
            throw new SnapshotException(this.metadata.name(), snapshotId, "failed to get snapshots", (Throwable)ex);
        }
    }

    @Override
    public MetaData getSnapshotGlobalMetaData(SnapshotId snapshotId) {
        try {
            return this.globalMetaDataFormat.read(this.blobContainer(), snapshotId.getUUID());
        }
        catch (NoSuchFileException ex) {
            throw new SnapshotMissingException(this.metadata.name(), snapshotId, (Throwable)ex);
        }
        catch (IOException ex) {
            throw new SnapshotException(this.metadata.name(), snapshotId, "failed to read global metadata", (Throwable)ex);
        }
    }

    @Override
    public IndexMetaData getSnapshotIndexMetaData(SnapshotId snapshotId, IndexId index) throws IOException {
        try {
            return this.indexMetaDataFormat.read(this.indexContainer(index), snapshotId.getUUID());
        }
        catch (NoSuchFileException e) {
            throw new SnapshotMissingException(this.metadata.name(), snapshotId, (Throwable)e);
        }
    }

    private BlobPath indicesPath() {
        return this.basePath().add("indices");
    }

    private BlobContainer indexContainer(IndexId indexId) {
        return this.blobStore().blobContainer(this.indicesPath().add(indexId.getId()));
    }

    private BlobContainer shardContainer(IndexId indexId, ShardId shardId) {
        return this.shardContainer(indexId, shardId.getId());
    }

    private BlobContainer shardContainer(IndexId indexId, int shardId) {
        return this.blobStore().blobContainer(this.indicesPath().add(indexId.getId()).add(Integer.toString(shardId)));
    }

    private RateLimiter getRateLimiter(Settings repositorySettings, String setting, ByteSizeValue defaultRate) {
        ByteSizeValue maxSnapshotBytesPerSec = repositorySettings.getAsBytesSize(setting, defaultRate);
        if (maxSnapshotBytesPerSec.getBytes() <= 0L) {
            return null;
        }
        return new RateLimiter.SimpleRateLimiter(maxSnapshotBytesPerSec.getMbFrac());
    }

    @Override
    public long getSnapshotThrottleTimeInNanos() {
        return this.snapshotRateLimitingTimeInNanos.count();
    }

    @Override
    public long getRestoreThrottleTimeInNanos() {
        return this.restoreRateLimitingTimeInNanos.count();
    }

    protected void assertSnapshotOrGenericThread() {
        assert (Thread.currentThread().getName().contains(SNAPSHOT_CODEC) || Thread.currentThread().getName().contains("generic")) : "Expected current thread [" + Thread.currentThread() + "] to be the snapshot or generic thread.";
    }

    @Override
    public String startVerification() {
        try {
            if (this.isReadOnly()) {
                this.latestIndexBlobId();
                return "read-only";
            }
            String seed = UUIDs.randomBase64UUID();
            byte[] testBytes = Strings.toUTF8Bytes(seed);
            BlobContainer testContainer = this.blobStore().blobContainer(this.basePath().add(BlobStoreRepository.testBlobPrefix(seed)));
            BytesArray bytes = new BytesArray(testBytes);
            try (StreamInput stream = bytes.streamInput();){
                testContainer.writeBlobAtomic("master.dat", stream, bytes.length(), true);
            }
            return seed;
        }
        catch (IOException exp) {
            throw new RepositoryVerificationException(this.metadata.name(), "path " + this.basePath() + " is not accessible on master node", exp);
        }
    }

    @Override
    public void endVerification(String seed) {
        if (!this.isReadOnly()) {
            try {
                String testPrefix = BlobStoreRepository.testBlobPrefix(seed);
                BlobContainer container = this.blobStore().blobContainer(this.basePath().add(testPrefix));
                container.deleteBlobsIgnoringIfNotExists(new ArrayList<String>(container.listBlobs().keySet()));
                this.blobStore().blobContainer(this.basePath()).deleteBlobIgnoringIfNotExists(testPrefix);
            }
            catch (IOException exp) {
                throw new RepositoryVerificationException(this.metadata.name(), "cannot delete test data at " + this.basePath(), exp);
            }
        }
    }

    @Override
    public RepositoryData getRepositoryData() {
        while (true) {
            long generation;
            try {
                generation = this.latestIndexBlobId();
            }
            catch (IOException ioe) {
                throw new RepositoryException(this.metadata.name(), "Could not determine repository generation from root blobs", ioe);
            }
            long genToLoad = this.latestKnownRepoGen.updateAndGet(known -> Math.max(known, generation));
            if (genToLoad > generation) {
                logger.info("Determined repository generation [" + generation + "] from repository contents but correct generation must be at least [" + genToLoad + "]");
            }
            try {
                return this.getRepositoryData(genToLoad);
            }
            catch (RepositoryException e) {
                if (genToLoad != this.latestKnownRepoGen.get()) {
                    logger.warn("Failed to load repository data generation [" + genToLoad + "] because a concurrent operation moved the current generation to [" + this.latestKnownRepoGen.get() + "]", (Throwable)e);
                    continue;
                }
                throw e;
            }
            break;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private RepositoryData getRepositoryData(long indexGen) {
        if (indexGen == -1L) {
            return RepositoryData.EMPTY;
        }
        try {
            String snapshotsIndexBlobName = "index-" + Long.toString(indexGen);
            try (InputStream blob = this.blobContainer().readBlob(snapshotsIndexBlobName);){
                RepositoryData repositoryData;
                block16: {
                    XContentParser parser = XContentType.JSON.xContent().createParser(NamedXContentRegistry.EMPTY, (DeprecationHandler)LoggingDeprecationHandler.INSTANCE, blob);
                    try {
                        repositoryData = RepositoryData.snapshotsFromXContent(parser, indexGen);
                        if (parser == null) break block16;
                    }
                    catch (Throwable throwable) {
                        if (parser != null) {
                            try {
                                parser.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    parser.close();
                }
                return repositoryData;
            }
        }
        catch (IOException ioe) {
            if (this.latestKnownRepoGen.compareAndSet(indexGen, -1L)) {
                logger.warn("Resetting repository generation tracker because we failed to read generation [" + indexGen + "]", (Throwable)ioe);
            }
            throw new RepositoryException(this.metadata.name(), "could not read repository data from index blob", ioe);
        }
    }

    private static String testBlobPrefix(String seed) {
        return TESTS_FILE + seed;
    }

    @Override
    public boolean isReadOnly() {
        return this.readOnly;
    }

    protected void writeIndexGen(RepositoryData repositoryData, long expectedGen) throws IOException {
        BytesReference genBytes;
        assert (!this.isReadOnly());
        long currentGen = repositoryData.getGenId();
        if (currentGen != expectedGen) {
            throw new RepositoryException(this.metadata.name(), "concurrent modification of the index-N file, expected current generation [" + expectedGen + "], actual current generation [" + currentGen + "] - possibly due to simultaneous snapshot deletion requests");
        }
        long newGen = currentGen + 1L;
        if (this.latestKnownRepoGen.get() >= newGen) {
            throw new IllegalArgumentException("Tried writing generation [" + newGen + "] but repository is at least at generation [" + newGen + "] already");
        }
        String indexBlob = "index-" + Long.toString(newGen);
        logger.debug("Repository [{}] writing new index generational blob [{}]", (Object)this.metadata.name(), (Object)indexBlob);
        this.writeAtomic(indexBlob, BytesReference.bytes(repositoryData.snapshotsToXContent(XContentFactory.jsonBuilder())), true);
        long latestKnownGen = this.latestKnownRepoGen.updateAndGet(known -> Math.max(known, newGen));
        if (newGen < latestKnownGen) {
            throw new IllegalStateException("Wrote generation [" + newGen + "] but latest known repo gen concurrently changed to [" + latestKnownGen + "]");
        }
        try (BytesStreamOutput bStream = new BytesStreamOutput();){
            bStream.writeLong(newGen);
            genBytes = bStream.bytes();
        }
        logger.debug("Repository [{}] updating index.latest with generation [{}]", (Object)this.metadata.name(), (Object)newGen);
        this.writeAtomic(INDEX_LATEST_BLOB, genBytes, false);
        if (newGen - 2L >= 0L) {
            String oldSnapshotIndexFile = "index-" + Long.toString(newGen - 2L);
            try {
                this.blobContainer().deleteBlobIgnoringIfNotExists(oldSnapshotIndexFile);
            }
            catch (IOException e) {
                logger.warn("Failed to clean up old index blob [{}]", (Object)oldSnapshotIndexFile);
            }
        }
    }

    long latestIndexBlobId() throws IOException {
        try {
            return this.listBlobsToGetLatestIndexId();
        }
        catch (UnsupportedOperationException e) {
            try {
                return this.readSnapshotIndexLatestBlob();
            }
            catch (NoSuchFileException nsfe) {
                return -1L;
            }
        }
    }

    long readSnapshotIndexLatestBlob() throws IOException {
        return Numbers.bytesToLong(Streams.readFully(this.blobContainer().readBlob(INDEX_LATEST_BLOB)).toBytesRef());
    }

    private long listBlobsToGetLatestIndexId() throws IOException {
        return this.latestGeneration(this.blobContainer().listBlobsByPrefix("index-").keySet());
    }

    private long latestGeneration(Collection<String> rootBlobs) {
        long latest = -1L;
        for (String blobName : rootBlobs) {
            if (!blobName.startsWith("index-")) continue;
            try {
                long curr = Long.parseLong(blobName.substring("index-".length()));
                latest = Math.max(latest, curr);
            }
            catch (NumberFormatException nfe) {
                logger.warn("[{}] Unknown blob in the repository: {}", (Object)this.metadata.name(), (Object)blobName);
            }
        }
        return latest;
    }

    private void writeAtomic(String blobName, BytesReference bytesRef, boolean failIfAlreadyExists) throws IOException {
        try (StreamInput stream = bytesRef.streamInput();){
            this.blobContainer().writeBlobAtomic(blobName, stream, bytesRef.length(), failIfAlreadyExists);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void snapshotShard(final Store store, MapperService mapperService, final SnapshotId snapshotId, final IndexId indexId, IndexCommit snapshotIndexCommit, final IndexShardSnapshotStatus snapshotStatus, ActionListener<Void> listener) {
        final ShardId shardId = store.shardId();
        long startTime = this.threadPool.absoluteTimeInMillis();
        try {
            Collection fileNames;
            Store.MetadataSnapshot metadataFromStore;
            Map<String, BlobMetaData> blobs;
            logger.debug("[{}] [{}] snapshot to [{}] ...", (Object)shardId, (Object)snapshotId, (Object)this.metadata.name());
            BlobContainer shardContainer = this.shardContainer(indexId, shardId);
            try {
                blobs = shardContainer.listBlobsByPrefix("index-");
            }
            catch (IOException e) {
                throw new IndexShardSnapshotFailedException(shardId, "failed to list blobs", e);
            }
            Tuple<BlobStoreIndexShardSnapshots, Long> tuple = this.buildBlobStoreIndexShardSnapshots(blobs.keySet(), shardContainer);
            BlobStoreIndexShardSnapshots snapshots = (BlobStoreIndexShardSnapshots)tuple.v1();
            long fileListGeneration = (Long)tuple.v2();
            if (snapshots.snapshots().stream().anyMatch(sf -> sf.snapshot().equals(snapshotId.getName()))) {
                throw new IndexShardSnapshotFailedException(shardId, "Duplicate snapshot name [" + snapshotId.getName() + "] detected, aborting");
            }
            ArrayList<BlobStoreIndexShardSnapshot.FileInfo> indexCommitPointFiles = new ArrayList<BlobStoreIndexShardSnapshot.FileInfo>();
            ArrayList<BlobStoreIndexShardSnapshot.FileInfo> filesToSnapshot = new ArrayList<BlobStoreIndexShardSnapshot.FileInfo>();
            store.incRef();
            try {
                try {
                    logger.trace("[{}] [{}] Loading store metadata using index commit [{}]", (Object)shardId, (Object)snapshotId, (Object)snapshotIndexCommit);
                    metadataFromStore = store.getMetadata(snapshotIndexCommit);
                    fileNames = snapshotIndexCommit.getFileNames();
                }
                catch (IOException e) {
                    throw new IndexShardSnapshotFailedException(shardId, "Failed to get store file metadata", e);
                }
            }
            finally {
                store.decRef();
            }
            int indexIncrementalFileCount = 0;
            int indexTotalNumberOfFiles = 0;
            long indexIncrementalSize = 0L;
            long indexTotalFileCount = 0L;
            for (String fileName : fileNames) {
                if (snapshotStatus.isAborted()) {
                    logger.debug("[{}] [{}] Aborted on the file [{}], exiting", (Object)shardId, (Object)snapshotId, (Object)fileName);
                    throw new IndexShardSnapshotFailedException(shardId, "Aborted");
                }
                logger.trace("[{}] [{}] Processing [{}]", (Object)shardId, (Object)snapshotId, (Object)fileName);
                StoreFileMetaData md = metadataFromStore.get(fileName);
                BlobStoreIndexShardSnapshot.FileInfo existingFileInfo = null;
                List<BlobStoreIndexShardSnapshot.FileInfo> filesInfo = snapshots.findPhysicalIndexFiles(fileName);
                if (filesInfo != null) {
                    for (BlobStoreIndexShardSnapshot.FileInfo fileInfo : filesInfo) {
                        if (!fileInfo.isSame(md)) continue;
                        existingFileInfo = fileInfo;
                        break;
                    }
                }
                indexTotalFileCount += md.length();
                ++indexTotalNumberOfFiles;
                if (existingFileInfo == null) {
                    ++indexIncrementalFileCount;
                    indexIncrementalSize += md.length();
                    final BlobStoreIndexShardSnapshot.FileInfo snapshotFileInfo = new BlobStoreIndexShardSnapshot.FileInfo(DATA_BLOB_PREFIX + UUIDs.randomBase64UUID(), md, this.chunkSize());
                    indexCommitPointFiles.add(snapshotFileInfo);
                    filesToSnapshot.add(snapshotFileInfo);
                    continue;
                }
                indexCommitPointFiles.add(existingFileInfo);
            }
            snapshotStatus.moveToStarted(startTime, indexIncrementalFileCount, indexTotalNumberOfFiles, indexIncrementalSize, indexTotalFileCount);
            assert (indexIncrementalFileCount == filesToSnapshot.size());
            StepListener allFilesUploadedListener = new StepListener();
            allFilesUploadedListener.whenComplete(v -> {
                List<String> blobsToDelete;
                IndexShardSnapshotStatus.Copy lastSnapshotStatus = snapshotStatus.moveToFinalize(snapshotIndexCommit.getGeneration());
                BlobStoreIndexShardSnapshot snapshot = new BlobStoreIndexShardSnapshot(snapshotId.getName(), lastSnapshotStatus.getIndexVersion(), indexCommitPointFiles, lastSnapshotStatus.getStartTime(), this.threadPool.absoluteTimeInMillis() - lastSnapshotStatus.getStartTime(), lastSnapshotStatus.getIncrementalFileCount(), lastSnapshotStatus.getIncrementalSize());
                logger.trace("[{}] [{}] writing shard snapshot file", (Object)shardId, (Object)snapshotId);
                try {
                    this.indexShardSnapshotFormat.write(snapshot, shardContainer, snapshotId.getUUID(), false);
                }
                catch (IOException e) {
                    throw new IndexShardSnapshotFailedException(shardId, "Failed to write commit point", e);
                }
                ArrayList<SnapshotFiles> newSnapshotsList = new ArrayList<SnapshotFiles>();
                newSnapshotsList.add(new SnapshotFiles(snapshot.snapshot(), snapshot.indexFiles()));
                for (SnapshotFiles point : snapshots) {
                    newSnapshotsList.add(point);
                }
                String indexGeneration = Long.toString(fileListGeneration + 1L);
                try {
                    BlobStoreIndexShardSnapshots updatedSnapshots = new BlobStoreIndexShardSnapshots(newSnapshotsList);
                    this.indexShardSnapshotsFormat.writeAtomic(updatedSnapshots, shardContainer, indexGeneration);
                    blobsToDelete = blobs.keySet().stream().filter(blob -> blob.startsWith("index-")).collect(Collectors.toList());
                    assert (blobsToDelete.stream().mapToLong(b -> Long.parseLong(b.replaceFirst("index-", ""))).max().orElse(-1L) < Long.parseLong(indexGeneration)) : "Tried to delete an index-N blob newer than the current generation [" + indexGeneration + "] when deleting index-N blobs " + blobsToDelete;
                }
                catch (IOException e) {
                    throw new IndexShardSnapshotFailedException(shardId, "Failed to finalize snapshot creation [" + snapshotId + "] with shard index [" + this.indexShardSnapshotsFormat.blobName(indexGeneration) + "]", e);
                }
                try {
                    shardContainer.deleteBlobsIgnoringIfNotExists(blobsToDelete);
                }
                catch (IOException e) {
                    logger.warn(() -> new ParameterizedMessage("[{}][{}] failed to delete old index-N blobs during finalization", (Object)snapshotId, (Object)shardId), (Throwable)e);
                }
                snapshotStatus.moveToDone(this.threadPool.absoluteTimeInMillis());
                listener.onResponse(null);
            }, listener::onFailure);
            if (indexIncrementalFileCount == 0) {
                allFilesUploadedListener.onResponse(Collections.emptyList());
                return;
            }
            final GroupedActionListener filesListener = new GroupedActionListener(allFilesUploadedListener, indexIncrementalFileCount);
            ExecutorService executor = this.threadPool.executor(SNAPSHOT_CODEC);
            final AtomicBoolean alreadyFailed = new AtomicBoolean();
            for (final BlobStoreIndexShardSnapshot.FileInfo snapshotFileInfo : filesToSnapshot) {
                executor.execute(new ActionRunnable<Void>(filesListener){

                    @Override
                    protected void doRun() {
                        try {
                            if (!alreadyFailed.get()) {
                                if (store.tryIncRef()) {
                                    try {
                                        BlobStoreRepository.this.snapshotFile(snapshotFileInfo, indexId, shardId, snapshotId, snapshotStatus, store);
                                    }
                                    finally {
                                        store.decRef();
                                    }
                                } else {
                                    if (snapshotStatus.isAborted()) {
                                        throw new IndexShardSnapshotFailedException(shardId, "Aborted");
                                    }
                                    assert (false) : "Store was closed before aborting the snapshot";
                                    throw new IllegalStateException("Store is closed already");
                                }
                            }
                            filesListener.onResponse(null);
                        }
                        catch (IOException e) {
                            throw new IndexShardSnapshotFailedException(shardId, "Failed to perform snapshot (index files)", e);
                        }
                    }

                    @Override
                    public void onFailure(Exception e) {
                        alreadyFailed.set(true);
                        super.onFailure(e);
                    }
                });
            }
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    @Override
    public void restoreShard(Store store, SnapshotId snapshotId, Version version, IndexId indexId, ShardId snapshotShardId, RecoveryState recoveryState) {
        ShardId shardId = store.shardId();
        try {
            final BlobContainer container = this.shardContainer(indexId, snapshotShardId);
            BlobStoreIndexShardSnapshot snapshot = this.loadShardSnapshot(container, snapshotId);
            SnapshotFiles snapshotFiles = new SnapshotFiles(snapshot.snapshot(), snapshot.indexFiles());
            new FileRestoreContext(this.metadata.name(), shardId, snapshotId, recoveryState, 4096){

                @Override
                protected InputStream fileInputStream(final BlobStoreIndexShardSnapshot.FileInfo fileInfo) {
                    SlicedInputStream dataBlobCompositeStream = new SlicedInputStream(fileInfo.numberOfParts()){

                        @Override
                        protected InputStream openSlice(long slice) throws IOException {
                            return container.readBlob(fileInfo.partName(slice));
                        }
                    };
                    return BlobStoreRepository.this.restoreRateLimiter == null ? dataBlobCompositeStream : new RateLimitingInputStream(dataBlobCompositeStream, BlobStoreRepository.this.restoreRateLimiter, BlobStoreRepository.this.restoreRateLimitingTimeInNanos::inc);
                }
            }.restore(snapshotFiles, store);
        }
        catch (Exception e) {
            throw new IndexShardRestoreFailedException(shardId, "failed to restore snapshot [" + snapshotId + "]", e);
        }
    }

    @Override
    public IndexShardSnapshotStatus getShardSnapshotStatus(SnapshotId snapshotId, Version version, IndexId indexId, ShardId shardId) {
        BlobStoreIndexShardSnapshot snapshot = this.loadShardSnapshot(this.shardContainer(indexId, shardId), snapshotId);
        return IndexShardSnapshotStatus.newDone(snapshot.startTime(), snapshot.time(), snapshot.incrementalFileCount(), snapshot.totalFileCount(), snapshot.incrementalSize(), snapshot.totalSize());
    }

    @Override
    public void verify(String seed, DiscoveryNode localNode) {
        this.assertSnapshotOrGenericThread();
        if (this.isReadOnly()) {
            try {
                this.latestIndexBlobId();
            }
            catch (IOException e) {
                throw new RepositoryVerificationException(this.metadata.name(), "path " + this.basePath() + " is not accessible on node " + localNode, e);
            }
        }
        BlobContainer testBlobContainer = this.blobStore().blobContainer(this.basePath().add(BlobStoreRepository.testBlobPrefix(seed)));
        try {
            BytesArray bytes = new BytesArray(seed);
            try (StreamInput stream = bytes.streamInput();){
                testBlobContainer.writeBlob("data-" + localNode.getId() + ".dat", stream, bytes.length(), true);
            }
        }
        catch (IOException exp) {
            throw new RepositoryVerificationException(this.metadata.name(), "store location [" + this.blobStore() + "] is not accessible on the node [" + localNode + "]", exp);
        }
        try (InputStream masterDat = testBlobContainer.readBlob("master.dat");){
            String seedRead = Streams.readFully(masterDat).utf8ToString();
            if (!seedRead.equals(seed)) {
                throw new RepositoryVerificationException(this.metadata.name(), "Seed read from master.dat was [" + seedRead + "] but expected seed [" + seed + "]");
            }
        }
        catch (NoSuchFileException e) {
            throw new RepositoryVerificationException(this.metadata.name(), "a file written by master to the store [" + this.blobStore() + "] cannot be accessed on the node [" + localNode + "]. This might indicate that the store [" + this.blobStore() + "] is not shared between this node and the master node or that permissions on the store don't allow reading files written by the master node", e);
        }
        catch (IOException e) {
            throw new RepositoryVerificationException(this.metadata.name(), "Failed to verify repository", e);
        }
    }

    public String toString() {
        return "BlobStoreRepository[[" + this.metadata.name() + "], [" + this.blobStore() + ']' + ']';
    }

    private ShardSnapshotMetaDeleteResult deleteShardSnapshot(RepositoryData repositoryData, IndexId indexId, ShardId snapshotShardId, SnapshotId snapshotId) throws IOException {
        Map<String, BlobMetaData> blobs;
        BlobContainer shardContainer = this.shardContainer(indexId, snapshotShardId);
        try {
            blobs = shardContainer.listBlobs();
        }
        catch (IOException e) {
            throw new IndexShardSnapshotException(snapshotShardId, "Failed to list content of shard directory", e);
        }
        Tuple<BlobStoreIndexShardSnapshots, Long> tuple = this.buildBlobStoreIndexShardSnapshots(blobs.keySet(), shardContainer);
        BlobStoreIndexShardSnapshots snapshots = (BlobStoreIndexShardSnapshots)tuple.v1();
        long fileListGeneration = (Long)tuple.v2();
        ArrayList<SnapshotFiles> newSnapshotsList = new ArrayList<SnapshotFiles>();
        Set survivingSnapshotNames = repositoryData.getSnapshots(indexId).stream().map(SnapshotId::getName).collect(Collectors.toSet());
        for (SnapshotFiles point : snapshots) {
            if (!survivingSnapshotNames.contains(point.snapshot())) continue;
            newSnapshotsList.add(point);
        }
        String indexGeneration = Long.toString(fileListGeneration + 1L);
        try {
            List<String> blobsToDelete;
            if (newSnapshotsList.isEmpty()) {
                blobsToDelete = new ArrayList<String>(blobs.keySet());
            } else {
                Set<String> survivingSnapshotUUIDs = repositoryData.getSnapshots(indexId).stream().map(SnapshotId::getUUID).collect(Collectors.toSet());
                BlobStoreIndexShardSnapshots updatedSnapshots = new BlobStoreIndexShardSnapshots(newSnapshotsList);
                this.indexShardSnapshotsFormat.writeAtomic(updatedSnapshots, shardContainer, indexGeneration);
                blobsToDelete = BlobStoreRepository.unusedBlobs(blobs, survivingSnapshotUUIDs, updatedSnapshots);
            }
            return new ShardSnapshotMetaDeleteResult(indexId, snapshotShardId.id(), blobsToDelete);
        }
        catch (IOException e) {
            throw new IndexShardSnapshotFailedException(snapshotShardId, "Failed to finalize snapshot deletion [" + snapshotId + "] with shard index [" + this.indexShardSnapshotsFormat.blobName(indexGeneration) + "]", e);
        }
    }

    private static List<String> unusedBlobs(Map<String, BlobMetaData> blobs, Set<String> survivingSnapshotUUIDs, BlobStoreIndexShardSnapshots updatedSnapshots) {
        return blobs.keySet().stream().filter(blob -> blob.startsWith("index-") || blob.startsWith(SNAPSHOT_PREFIX) && blob.endsWith(".dat") && !survivingSnapshotUUIDs.contains(blob.substring(SNAPSHOT_PREFIX.length(), blob.length() - ".dat".length())) || blob.startsWith(DATA_BLOB_PREFIX) && updatedSnapshots.findNameFile(BlobStoreIndexShardSnapshot.FileInfo.canonicalName(blob)) == null || FsBlobContainer.isTempBlobName(blob)).collect(Collectors.toList());
    }

    private BlobStoreIndexShardSnapshot loadShardSnapshot(BlobContainer shardContainer, SnapshotId snapshotId) {
        try {
            return this.indexShardSnapshotFormat.read(shardContainer, snapshotId.getUUID());
        }
        catch (NoSuchFileException ex) {
            throw new SnapshotMissingException(this.metadata.name(), snapshotId, (Throwable)ex);
        }
        catch (IOException ex) {
            throw new SnapshotException(this.metadata.name(), snapshotId, "failed to read shard snapshot file for [" + shardContainer.path() + ']', (Throwable)ex);
        }
    }

    private Tuple<BlobStoreIndexShardSnapshots, Long> buildBlobStoreIndexShardSnapshots(Set<String> blobs, BlobContainer shardContainer) throws IOException {
        long latest = this.latestGeneration(blobs);
        if (latest >= 0L) {
            BlobStoreIndexShardSnapshots shardSnapshots = this.indexShardSnapshotsFormat.read(shardContainer, Long.toString(latest));
            return new Tuple((Object)shardSnapshots, (Object)latest);
        }
        if (blobs.stream().anyMatch(b -> b.startsWith(SNAPSHOT_PREFIX) || b.startsWith("index-") || b.startsWith(DATA_BLOB_PREFIX))) {
            throw new IllegalStateException("Could not find a readable index-N file in a non-empty shard snapshot directory [" + shardContainer.path() + "]");
        }
        return new Tuple((Object)BlobStoreIndexShardSnapshots.EMPTY, (Object)latest);
    }

    private void snapshotFile(final BlobStoreIndexShardSnapshot.FileInfo fileInfo, IndexId indexId, final ShardId shardId, final SnapshotId snapshotId, final IndexShardSnapshotStatus snapshotStatus, Store store) throws IOException {
        BlobContainer shardContainer = this.shardContainer(indexId, shardId);
        String file = fileInfo.physicalName();
        try (IndexInput indexInput = store.openVerifyingInput(file, IOContext.READONCE, fileInfo.metadata());){
            int i = 0;
            while ((long)i < fileInfo.numberOfParts()) {
                long partBytes = fileInfo.partBytes(i);
                InputStream inputStream = new InputStreamIndexInput(indexInput, partBytes);
                if (this.snapshotRateLimiter != null) {
                    inputStream = new RateLimitingInputStream(inputStream, this.snapshotRateLimiter, this.snapshotRateLimitingTimeInNanos::inc);
                }
                inputStream = new FilterInputStream(inputStream){

                    @Override
                    public int read() throws IOException {
                        this.checkAborted();
                        return super.read();
                    }

                    @Override
                    public int read(byte[] b, int off, int len) throws IOException {
                        this.checkAborted();
                        return super.read(b, off, len);
                    }

                    private void checkAborted() {
                        if (snapshotStatus.isAborted()) {
                            logger.debug("[{}] [{}] Aborted on the file [{}], exiting", (Object)shardId, (Object)snapshotId, (Object)fileInfo.physicalName());
                            throw new IndexShardSnapshotFailedException(shardId, "Aborted");
                        }
                    }
                };
                shardContainer.writeBlob(fileInfo.partName(i), inputStream, partBytes, true);
                ++i;
            }
            Store.verify(indexInput);
            snapshotStatus.addProcessedFile(fileInfo.length());
        }
        catch (Exception t) {
            BlobStoreRepository.failStoreIfCorrupted(store, t);
            snapshotStatus.addProcessedFile(0L);
            throw t;
        }
    }

    private static void failStoreIfCorrupted(Store store, Exception e) {
        if (Lucene.isCorruptionException(e)) {
            try {
                store.markStoreCorrupted((IOException)e);
            }
            catch (IOException inner) {
                inner.addSuppressed(e);
                logger.warn("store cannot be marked as corrupted", (Throwable)inner);
            }
        }
    }

    private static final class ShardSnapshotMetaDeleteResult {
        private final IndexId indexId;
        private final int shardId;
        private final Collection<String> blobsToDelete;

        ShardSnapshotMetaDeleteResult(IndexId indexId, int shardId, Collection<String> blobsToDelete) {
            this.indexId = indexId;
            this.shardId = shardId;
            this.blobsToDelete = blobsToDelete;
        }
    }
}

